#!/usr/bin/perl -w

# Livestat 1.2, by Jason Weill (livestat@weill.org)
# Copyright (C) 2001-02 Jason Weill

# Livestat is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# Livestat is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Livestat; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# Generates pages based on individual statistics.

my %options;
my %teams;

if (not -e "activetourney") {
  die "No tournament is active.  Use set-active to set a tournament.";
}

# read the HTML header and footer
my $page_header = "";
my $page_footer = "";

if (not -e "common.head") {
  die "Header file not found";
}

if (not -e "common.foot") {
  die "Footer file not found";
}

open(HEADER, "< common.head") or die "Cannot open common header: $!";
foreach (<HEADER>) {
  $page_header .= $_;
}
close(HEADER) or die "Cannot close common header: $!";

open(FOOTER, "< common.foot") or die "Cannot open common footer: $!";
foreach (<FOOTER>) {
  $page_footer .= $_;
}
close(FOOTER) or die "Cannot close common footer: $!";

open ACTIVE, "activetourney" or die "Cannot open tournament file: $!";
chomp($dir = <ACTIVE>);
close ACTIVE or die "Cannot close tournament file: $!";

chdir($dir) or die "Cannot open directory $dir: $!";

open CONFIG, "global.cfg" or die "Cannot open configuration file: $!";

$_ = "";
until (/BEGIN TEAM/) {
  $_ = <CONFIG>;
  chomp;
  #s/\#.*//g;
  if (/=/ and not /^\#/) {
    ($option, $setting) = split(/=/, $_);
    $options{$option} = $setting;
  }
}

until (/END TEAM/) {
  $_ = <CONFIG>;
  chomp;
  if (/=/ and not /^\#/) {
    ($option, $setting) = split(/=/, $_);
    $teams{$option} = $setting;
  }
}

close CONFIG or die "Cannot close config file: $!";

open STATS, "indstats" or die "Cannot open stats file: $!";
my %records;

my $rounds = 0;

while(not eof(STATS)) {
  chomp($_ = <STATS>);
  if (/,/) {
    @sets = split(/,/, $_);
    ($round, $team1, $name, $heard, $pheard, $power, $tossup, $neg5) =
      @sets[0,1,3,4,5,6,7,8];

    $pname = "$name, " . $teams{$team1};

    if (not $records{$pname}) {
      $records{$pname} = { games => 0, pheard => 0, points => 0, fifteens => 0,
			   tens => 0, negs => 0 };
    }

    if ($round > $rounds) { $rounds = $round; }

    $records{$pname}{"games"} += $pheard / $heard;
    $records{$pname}{"pheard"} += $pheard;
    $records{$pname}{"points"} += (15 * $power + 10 * $tossup - 5 * $neg5);
    $records{$pname}{"fifteens"} += $power;
    $records{$pname}{"tens"} += $tossup;
    $records{$pname}{"negs"} += $neg5;
  }
}

sub by_points {
  # Defaults to PPG, then total points (i.e. games played), then fewest negs.
  $ppg_a = $records{$a}{"points"} / $records{$a}{"games"};  
  $ppg_b = $records{$b}{"points"} / $records{$b}{"games"};

  if ($ppg_a == $ppg_b) {
    if ($records{$a}{"points"} == $records{$b}{"points"}) {
      if ($records{$a}{"negs"} == $records{$b}{"negs"}) {
	return $a cmp $b; # sort by name, worst case
      } else {
	return $records{$a}{"negs"} <=> $records{$b}{"negs"};
      }
    } else {
      return $records{$a}{"points"} <=> $records{$b}{"points"};
    }
  } else {
    return $ppg_a <=> $ppg_b;
  }
}

sub by_negs {
  # Sorts by number of neg5's, followed by -5PG, lowest toss-up to neg ratio,
  # and then name.
  $ppg_a = $records{$a}{"negs"} / $records{$a}{"games"};  
  $ppg_b = $records{$b}{"negs"} / $records{$b}{"games"};
  $tun_a = ($records{$a}{"negs"} == 0 ? 99.99 : 
	    ($records{$a}{"fifteens"} + $records{$a}{"tens"}) / 
	    $records{$a}{"negs"});
  $tun_b = ($records{$b}{"negs"} == 0 ? 99.99 : 
	    ($records{$b}{"fifteens"} + $records{$b}{"tens"}) / 
	    $records{$b}{"negs"});

  if ($records{$a}{"negs"} == $records{$b}{"negs"}) {
    if ($ppg_a == $ppg_b) {
      if ($tun_a == $tun_b) {
	return $a cmp $b;
      } else {
	return $tun_b <=> $tun_a; # ascending sort
      }
    } else {
      return $ppg_a <=> $ppg_b; 
    }
  } else {
    return $records{$a}{"negs"} <=> $records{$b}{"negs"};
  }
}

@sortedplyrs = sort by_points keys(%records);

open STANDINGS, ">personal.html" or die "Cannot open output file: $!";

$title = $options{"name"};
$page_header =~ s/TITLEHERE/$title Individual Standings/;
print STANDINGS $page_header;
print STANDINGS "<h1>$title</h1>\n";
print STANDINGS "<h2>Standings after " . $rounds . " round" . 
  ($rounds == 1 ? "" : "s") . "</h2>\n";
  print STANDINGS 
'<p>Standings: <a href="stats.html">By Team</a> - 
<strong>By Individual</strong> (<a href="#neg">Top Neg-5</a>) - 
<a href="rounds.html">By Round</a></p>';
print STANDINGS "<table class='standings'>\n";
print STANDINGS "<tr>\n" .
  "<th colspan='2'>PPG Leaders</th>\n" .
  "<th class='number'>GP</th>\n";

if ($options{"power"} eq "y") {
  print STANDINGS "<th class='number'>+15</th>\n";
}

print STANDINGS "<th class='number'>+10</th>\n" .
  "<th class='number'>-5</th>\n" .
  "<th class='number'>Pts</th>\n" .
  "<th class='number'>PPG</th>\n";

if ($options{"timed"} eq "y") {
  print STANDINGS "<th class='number'>PP20H</th>\n";
}

print STANDINGS "<th class='number'>TU/-5</th>\n" . 
  "</tr>";
#foreach $key(keys(%records)) {
my $rank = 0;
foreach $key(reverse @sortedplyrs) {
  $rank++;
  print STANDINGS "<tr>\n";
  print STANDINGS "<td>$rank.</td>\n";
  print STANDINGS "<td>" . $key . "</td>\n";
  printf STANDINGS "<td class='number'>%01.1f</td>\n",  
    $records{$key}{"games"};

  if ($options{"power"} eq "y") {
    print STANDINGS "<td class='number'>" . $records{$key}{"fifteens"} . 
      "</td>\n";
  }

  print STANDINGS "<td class='number'>" . $records{$key}{"tens"} . "</td>\n";
  print STANDINGS "<td class='number'>" . $records{$key}{"negs"} . "</td>\n";
  print STANDINGS "<td class='number'>" . $records{$key}{"points"} . "</td>\n";
  printf STANDINGS "<td class='number'>%01.2f</td>\n", 
  ($records{$key}{"points"} / $records{$key}{"games"});
  
  if ($options{"timed"} eq "y") {
    printf STANDINGS "<td class='number'>%01.2f</td>\n",
    ($records{$key}{"points"} / $records{$key}{"pheard"}) * 20;
  }
  
  if ($records{$key}{"negs"} == 0) {
    print STANDINGS "<td class='number'>N/A</td>\n";
  } else {
    printf STANDINGS "<td class='number'>%01.2f</td>\n", 
    ($records{$key}{"fifteens"} + $records{$key}{"tens"}) 
      / $records{$key}{"negs"};
  }

  print STANDINGS "</tr>";
}
  
print STANDINGS "</table>";

# Neg-5 leaders table
#print STANDINGS "<h3><a name='neg'>Top Neg-5ers</a></h3>\n";

@sortedplyrs = sort by_negs keys(%records);
print STANDINGS "<table class='standings'>\n";
print STANDINGS "<tr>\n" .
  "<th colspan='2'><a name='neg'>Top Neg-5ers</a></th>\n" .
  "<th class='number'>GP</th>\n" . 
  "<th class='number'>-5</th>\n" . 
  "<th class='number'>-5PG</th>\n" .
  "<th class='number'>TU/-5</th>\n" .
  "</tr>\n";

# my $rank = 0;
my $minneg = 3;
if ($minneg > $#sortedplyrs) {
  $minneg = $#sortedplyrs;
}
for ($rank = 1; $rank <= $minneg; $rank++) {
  $key = $sortedplyrs[-$rank];
  print STANDINGS "<tr>\n";
  print STANDINGS "<td>$rank.</td>\n";
  print STANDINGS "<td>" . $key . "</td>\n";
  printf STANDINGS "<td class='number'>%01.1f</td>\n",  
    $records{$key}{"games"};
  print STANDINGS "<td class='number'>" . $records{$key}{"negs"} . "</td>\n";
  printf STANDINGS "<td class='number'>%01.2f</td>\n", 
  ($records{$key}{"negs"} / $records{$key}{"games"});
  
  if ($records{$key}{"negs"} == 0) {
    print STANDINGS "<td class='number'>N/A</td>\n";
  } else {
    printf STANDINGS "<td class='number'>%01.2f</td>\n", 
    ($records{$key}{"fifteens"} + $records{$key}{"tens"}) 
      / $records{$key}{"negs"};
  }

  print STANDINGS "</tr>";
}
print STANDINGS "</table>\n";

print STANDINGS $page_footer;
close STANDINGS or die "Cannot close output file: $!";
