#!/usr/bin/perl -w

# Livestat 1.2, by Jason Weill (livestat@weill.org)
# Copyright (C) 2001-02 Jason Weill

# Livestat is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# Livestat is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with Livestat; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

# Generates pages based on individual statistics, by team.

my %options;
my %teams;

# read the HTML header and footer
my $page_header = "";
my $page_footer = "";

if (not -e "common.head") {
  die "Header file not found";
}

if (not -e "common.foot") {
  die "Footer file not found";
}

open(HEADER, "< common.head") or die "Cannot open common header: $!";
foreach (<HEADER>) {
  $page_header .= $_;
}
close(HEADER) or die "Cannot close common header: $!";

open(FOOTER, "< common.foot") or die "Cannot open common footer: $!";
foreach (<FOOTER>) {
  $page_footer .= $_;
}
close(FOOTER) or die "Cannot close common footer: $!";

if (not -e "activetourney") {
  die "No tournament is active.  Use set-active to set a tournament.";
}

open ACTIVE, "activetourney" or die "Cannot open tournament file: $!";
chomp($dir = <ACTIVE>);
close ACTIVE or die "Cannot close tournament file: $!";

chdir($dir) or die "Cannot open directory $dir: $!";

open CONFIG, "global.cfg" or die "Cannot open configuration file: $!";

$_ = "";
until (/BEGIN TEAM/) {
  $_ = <CONFIG>;
  chomp;
  #s/\#.*//g;
  if (/=/ and not /^\#/) {
    ($option, $setting) = split(/=/, $_);
    $options{$option} = $setting;
  }
}

until (/END TEAM/) {
  $_ = <CONFIG>;
  chomp;
  if (/=/ and not /^\#/) {
    ($option, $setting) = split(/=/, $_);
    $teams{$option} = $setting;
  }
}

close CONFIG or die "Cannot close config file: $!";

open STATS, "indstats" or die "Cannot open individual stats file: $!";
my %records;
my $points;
my $rounds = 0;
my %hround;
my %hr_set;

while(not eof(STATS)) {
  chomp($_ = <STATS>);
  if (/,/) {
    ($round, $team1, $team2, $name, $heard, $pheard, $power, $tossup, $neg5) =
      split(/,/, $_);

    if (!($hround{$team1})) {
      $hround{$team1} = 0;
    }

    if (!($hr_set{$team1}{$round})) {
      $hround{$team1} += $heard;
      $hr_set{$team1}{$round} = 1;
    }

    $pname = "$name, " . $teams{$team1};

    if (not $records{$pname}) {
      $records{$pname} = { name => "", team => "", 
			   games => 0, pheard => 0, points => 0, fifteens => 0,
			   tens => 0, negs => 0, bestgame_score => 0, 
			   bestgame_opp => "N/A", bestgame_round => 0};
    }

    if ($round > $rounds) { $rounds = $round; }

    $points = 15 * $power + 10 * $tossup - 5 * $neg5;
    $records{$pname}{"name"} = $name;
    $records{$pname}{"team"} = $team1;
    $records{$pname}{"games"} += $pheard / $heard;
    $records{$pname}{"pheard"} += $pheard;
    $records{$pname}{"points"} += $points;
    $records{$pname}{"fifteens"} += $power;
    $records{$pname}{"tens"} += $tossup;
    $records{$pname}{"negs"} += $neg5;
    if ($points > $records{$pname}{"bestgame_score"}) {
      $records{$pname}{"bestgame_score"} = $points;
      $records{$pname}{"bestgame_opp"} = $teams{$team2};
      $records{$pname}{"bestgame_round"} = $round;
    }
  }
}

close STATS;

my %games;
open STATS, "scores" or die "Cannot open scores file: $!";

while(not eof(STATS)) {
  chomp($_ = <STATS>);
  if (/,/) {
    ($round, $team1, $team2, $score1, $score2) = split(/,/, $_);

    if (not $games{$team1}) {
      $games{$team1} = ();
    }
    if (not $games{$team2}) {
      $games{$team2} = ();
    }

    my @tarray1 = ($round, $team2, $score1, $score2);
    my @tarray2 = ($round, $team1, $score2, $score1);

    push(@{$games{$team1}}, @tarray1);
    push(@{$games{$team2}}, @tarray2);
  }
}

close STATS;

sub by_teams {
  # Sort by teams, then names.
  if ($records{$a}{"team"} eq $records{$b}{"team"}) {
    return $records{$a}{"name"} cmp $records{$b}{"name"}; # sort by name
  } else {
    return $records{$a}{"team"} cmp $records{$b}{"team"};
  }
}

#sub by_round {
  # Sort by round.
#  return $games{$a}[0] <=> $games{$b}[0];
#}

@sortedplyrs = sort by_teams keys(%records);
#@sortedrounds = sort keys(%games);
my $curteam = "";

foreach $key(@sortedplyrs) {
  #print "looking at " . $key . " on " . $records{$key}{"team"} . "\n";
  if ($records{$key}{"team"} ne $curteam) {
    if (fileno(STANDINGS)) {
      print STANDINGS "</table>";
      print STANDINGS $page_footer;
      close STANDINGS or die "Cannot close output file: $!";
    }

    $curteam = $records{$key}{"team"};
    $filename = "stats." . $records{$key}{"team"} . ".html";
    open STANDINGS, ">$filename" or die "Cannot write $filename: $!";
  
    $myteam = $teams{$curteam};
    $title = $options{"name"};
    $myheader = $page_header;
    $myheader =~ s/TITLEHERE/$title Stats for $myteam/;
    print STANDINGS $myheader;
    print STANDINGS "<h1>$title</h1>\n";
    print STANDINGS "<h2>Standings after " . $rounds . " round" . 
      ($rounds == 1 ? "" : "s") . " for " . $myteam . "</h2>\n";
    print STANDINGS 
      '<p>Overall statistics: <a href="stats.html">By Team</a> - 
<a href="personal.html">By Individual</a> - 
<a href="rounds.html">By Round</a></p>';

    if ($options{"timed"} eq "y") {
      print STANDINGS
	"\n<p>This team has heard <strong>" . 
	  $hround{$curteam} . "</strong> toss-ups.</p>\n";
    }

    print STANDINGS "<h3>Match Summary</h3>\n";
    print STANDINGS "<table class='standings'>\n";
    print STANDINGS "<th>Round</th><th>Opponent</th><th>Result</th>\n";

    my ($wins, $losses, $ties) = (0, 0, 0);
    #my $elecount = 0;
    while (@{$games{$curteam}}) {
      $rd = shift(@{$games{$curteam}});
      $opp = shift(@{$games{$curteam}});
      $mys = shift(@{$games{$curteam}});
      $ops = shift(@{$games{$curteam}});
      #print $elecount++ . " ";
      #print $arr . "\n";
      print STANDINGS "<tr>\n";
      print STANDINGS "<td>" . $rd . "</td>\n";
      print STANDINGS "<td>" . $teams{$opp} . "</td>\n";
      print STANDINGS "<td>";
      if ($mys > $ops) {
	print STANDINGS "W";
	$wins++;
      } elsif ($ops > $mys) {
	print STANDINGS "L";
	$losses++;
      } else {
	print STANDINGS "T";
	$ties++;
      }
      print STANDINGS ", " . ($mys < 0 ? "($mys)" : "$mys") . "-" . 
	($ops < 0 ? "($ops)" : "$ops") . "</td>\n";
      print STANDINGS "</tr>";
    }
    print STANDINGS "<tr><td colspan='3'><strong>Overall:</strong>\n";
    printf STANDINGS "$wins-$losses-$ties (%01.3f)</td>\n</tr>\n</table>\n", 
    ($wins + 0.5 * $ties) / ($wins + $losses + $ties);

    print STANDINGS "<h3>Player Performance</h3>\n";
    print STANDINGS "<table class='standings'>\n";
    print STANDINGS "<tr>\n" .
      "<th>&nbsp;</th>\n" .
	"<th class='number'>GP</th>\n";
    
    if ($options{"power"} eq "y") {
      print STANDINGS "<th class='number'>+15</th>\n";
    }
    
    print STANDINGS "<th class='number'>+10</th>\n" .
      "<th class='number'>-5</th>\n" .
	"<th class='number'>Pts</th>\n" .
	  "<th class='number'>PPG</th>\n";
    
    if ($options{"timed"} eq "y") {
      print STANDINGS "<th class='number'>PP20H</th>\n";
    }
    
    print STANDINGS "<th class='number'>TU/-5</th>\n" . 
      "<th>Best Round</th>\n" .
	"</tr>";
    #foreach $key(keys(%records)) {
    #my $rank = 0;
  }
  
  #$rank++;
  print STANDINGS "<tr>\n";
  #print STANDINGS "<td>$rank.</td>\n";
  print STANDINGS "<td>" . $records{$key}{"name"} . "</td>\n";
  printf STANDINGS "<td class='number'>%01.1f</td>\n",  
    $records{$key}{"games"};

  if ($options{"power"} eq "y") {
    print STANDINGS "<td class='number'>" . $records{$key}{"fifteens"} . 
      "</td>\n";
  }

  print STANDINGS "<td class='number'>" . $records{$key}{"tens"} . "</td>\n";
  print STANDINGS "<td class='number'>" . $records{$key}{"negs"} . "</td>\n";
  print STANDINGS "<td class='number'>" . $records{$key}{"points"} . "</td>\n";
  printf STANDINGS "<td class='number'>%01.2f</td>\n", 
    ($records{$key}{"points"} / $records{$key}{"games"});
  
  if ($options{"timed"} eq "y") {
    printf STANDINGS "<td class='number'>%01.2f</td>\n",
      ($records{$key}{"points"} / $records{$key}{"pheard"}) * 20;
  }
  
  if ($records{$key}{"negs"} == 0) {
    print STANDINGS "<td class='number'>N/A</td>\n";
  } else {
    printf STANDINGS "<td class='number'>%01.2f</td>\n", 
      ($records{$key}{"fifteens"} + $records{$key}{"tens"}) 
	/ $records{$key}{"negs"};
  }

  if ($records{$key}{"bestgame_round"} == 0) {
    print STANDINGS "<td>N/A</td>\n";
  } else {
    print STANDINGS "<td>Round " . 
      $records{$key}{"bestgame_round"} . 
	": " . $records{$key}{"bestgame_score"} . " points against " . 
	  $records{$key}{"bestgame_opp"} . "</td>\n";
  }

  print STANDINGS "</tr>";
}
  
print STANDINGS "</table>";

print STANDINGS $page_footer;
close STANDINGS or die "Cannot close output file: $!";
